/************************************************************************
 * NAME:	ados.h
 *
 * DESCR:	Definitions of structures for the Apple Dos 3.3 Filesystem.
 *
 * NOTES:	- Most of the information used in this implemention come
 *		  from "Beneath Apple Dos" by Don Worth and Pieter Lechner.
 ************************************************************************/
/************************************************************************
/*
/*   Structure of Dos 3.3 on the diskette looks like this
/*
/*          Track
/*                ._______________________________.
/*            0   |                               |
/*            1   | Boot and Dos - tracks 0, 1, 2 |
/*            2   |_______________________________|
/*            3   |                               |
/*            .   |                               |
/*            .   | Files - tracks 3-16           |
/*            .   |                               |
/*           16   |_______________________________|
/*           17   | VTOC and Catalog track 17     |
/*           18   |-------------------------------|
/*            .   |                               |
/*            .   | Files - tracks 18-34          |
/*            .   |                               |
/*           34   |_______________________________|
/*
/*   VTOC - Volume Table of Contents - structure - Sector 0 - Track 17
/*
/*       Byte	Description
/*      -----	-------------------------------------------------------
/*      00	<Unused>
/*	01	Track number of first Catalog sector
/*	02	Sector number of first Catalog sector
/*	03	Release number of this DOS (03 - Dos 3.3)
/*	04-05	<Unused>
/*	06	Diskette volume number (1-254)
/*	07-26	<Unused>
/*	27	Maximum number of track/sector pairs int a track/sector list
/*		(122 for 256-byte sectors)
/*	28-2F	<Unused>
/*	30	Last track where sectors were allocated
/*	31	Direction for track allocation (+1 or -1)
/*	32-33	<Unused>
/*	34	Number of tracks per diskette (normally 35)
/*	35	Number of sectors per diskette (13 or 16)
/*	36-37	Number of bytes per sector (LO/HI format)
/*	38-3B	Bitmap of free sectors in Track 0	
/*	3C-3F	Bitmap of free sectors in Track 1
/*      ...	(format for bitmaps -  byte 0 - sectors FEDC BA98
/*		                       byte 1 - sectors 7654 3210
/*				       byte 2 + 3 - <Unused>
/*	BC-BF	Bitmap of free sectors in Track 33
/*	C0-C3	Bitmap of free sectors in Track 34
/*	C4-FF	Bitmaps for additional tracks
/*		 
/*
/*         
/*	
/*
/*
/*************************************************************************/

/************************************************************************
 * NAME:	STRUCT ts_ref
 *
 * DESCR:	A track/sector reference.  Pretty simple structure.
 ************************************************************************/

typedef struct {
    int		track;
    int		sector;
} ados_ts_ref;

/************************************************************************
 * NAME:	STRUCT vtoc
 *
 * DESCR:	The volume table of contents structure.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- the definition of the VTOC is NOT meant to be a bitmap of
 *		  the sector.  Instead, marshalling routines must be used to
 *		  read/write the VTOC sector based upon this structure.
 *		- allocation is an array that is used to tell
 *		  if a given sector is allocated (used) or not.
 *		  if allocation[track*sectors+sector] is TRUE, the sector is in use.
 ************************************************************************/
typedef struct {
    ados_ts_ref	catalog;	/* pointer to first catalog entry	*/
    int		release;	/* DOS release number			*/
    int		volume;		/* diskette volume number		*/
    int		max_ts;		/* maximum number of ts_refs		*/
    int		last_track;	/* last track with allocated sectors	*/
    int		direction;	/* direction of track allocation	*/
    int		tracks;		/* number of tracks on diskette		*/
    int		sectors;	/* number of sectors for each track	*/
    int		sector_size;	/* size of the sectors on the disk	*/
    int		*allocation;	/* array of free/used sectors		*/
} ados_vtoc;

#define ALLOCATION(v,t,s)	((v)->allocation[(t)*((v)->sectors)+(s)])
#define ALLOCATED(v,t,s)	(ALLOCATION(v,t,s) == TRUE)
#define ALLOCATION_SET(v,t,s)	(ALLOCATION(v,t,s) = TRUE)
#define ALLOCATION_CLEAR(v,t,s)	(ALLOCATION(v,t,s) = FALSE)

/************************************************************************
 * NAME:	ENUM file_type
 *
 * DESCR:	The type of file;
 *
 * NOTES:	
 ************************************************************************/
typedef enum {
    ADOS_FILE_TEXT,
    ADOS_FILE_IBASIC,
    ADOS_FILE_ABASIC,
    ADOS_FILE_BIN,
    ADOS_FILE_S,
    ADOS_FILE_R,
    ADOS_FILE_A,
    ADOS_FILE_B
} ados_file_type;

#define ADOS_FILE_NAME_LENGTH	30

/************************************************************************
 * NAME:	STRUCT file_entry
 *
 * DESCR:	A file in the DOS3.3 filesystem.
 *
 * NOTES:	- much like with the VTOC, this is NOT meant to be a bitmap
 *		  of a file entry.  Marshalling routines must be used to
 *		  read or write these things.
 ************************************************************************/
typedef struct {
    ados_ts_ref		ts;			/* ptr to first track/sect list	*/
    ados_file_type	type;			/* type of the file		*/
    int			locked;			/* TRUE if the file is locked	*/
    char		name[ADOS_FILE_NAME_LENGTH];	/* file name		*/
    int			length;			/* length in sectors		*/
    int			deleted;		/* if true, is a deleted file	*/
} ados_file_entry;

/************************************************************************
 * NAME:	STRUCT catalog
 *
 * DESCR:	The diskette catalog.
 *
 * NOTES:	- much like with the VTOC, this is NOT meant to be a bitmap
 *		  of a catalog entry.  Marshalling routines must be used to
 *		  read or write these things.
 *		- due to the nature of catalog entries, this structure
 *		  DOES NOT necessarily maintain the order of the entries
 *		  when reading or writing them.  However, the linkage is
 *		  done correctly in any case.
 *		- this routine counts the number of sectors of catalog
 *		  entries so that other routines won't add more entries
 *		  than were originally accounted for in the catalog.
 *		  This is NOT necessary, because the catalog could be
 *		  expanded with more sectors.  It is just a simplifying
 *		  assumption for this code.
 ************************************************************************/
typedef struct {
    int			 sectors;	/* number of sectors of catalog entries	*/
    int			 count;		/* current allocated file_entries	*/
    int			 index;		/* current used file_entries		*/
    ados_file_entry	*files;		/* an array of files			*/
} ados_catalog;

typedef struct {
    int			index;
} ados_catalog_cursor;

/************************************************************************
 * NAME:	ENUM adosfs_type
 *
 * DESCR:	What type of Apple DOS filesystem.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
typedef enum {
    DOS33,
    DOS32,
    PRODOS
} adosfs_type;

#define VTOC_TRACK		17
#define VTOC_SECTOR		0
#define VTOC_SIZE		256
#define CATALOG_SECTOR_SIZE	256
#define FILE_ENTRY_SIZE		35

/* definitions for abstracting the floppy structure	*/

typedef int (*adosreadfn)(void *,int,int,int,char*);
typedef int (*adoswritefn)(void *,int,int,int,char*);
typedef void *adosdevice;
    

/************************************************************************
 * NAME:	STRUCT adosfs
 *
 * DESCR:	Definition of the ADOS file structure.  Can hold any
 *		Apple Dos filesystem...though only 3.3 is implemented now.
 *
 * NOTES:	
 ************************************************************************/

struct ados_inode;

struct adosfs {
    adosfs_type		type;		/* type of ADOS file system			*/

    adosdevice		device;
    adosreadfn		preadsec;	/* routine for interfacing to the floppy struct	*/
    adoswritefn		pwritesec;	/* routine for interfacing to the floppy struct	*/
    int			startsec;	/* 0 or 1 for the phys start sector	*/

    ados_vtoc		*vtoc;		/* volume table of contents for the filesystem	*/
    ados_catalog	*catalog;	/* files for this filesystem			*/

    int			 filecount;
    struct ados_file	*filebuf;	/* file handle storage space		*/
};

#include "tslist.h"
#include "fileio.h"
#include "physio.h"
#include "logio.h"
#include "catalog.h"

/*************************************************************************/
/*************************************************************************/
/*******   EXTERNAL DEFINITIONS   ****************************************/
/*************************************************************************/

int ados33_init(struct adosfs *, adosdevice, adosreadfn, adoswritefn);
int ados33_cleanup(struct adosfs *);
int ados33_settings(struct adosfs *, int, int);
